<?php

class DMS
{
    /**
     * Holds freemius instance
     *
     * @since  1.0.0
     * @access private
     * @var Freemius $dms_fs we store the freemius instance
     */
    public  $dms_fs ;
    /**
     * Plugin base name
     *
     * @since  1.4.6
     * @var string $plugin_base_name
     */
    public  $plugin_base_name ;
    /**
     * Plugin dir
     *
     * @since  1.4.6
     * @var string $plugin_dir
     */
    public  $plugin_dir ;
    /**
     * Plugin url
     *
     * @since  1.4.6
     * @var string $plugin_url
     */
    public  $plugin_url ;
    /**
     * Plugin version
     *
     * @since  1.4.6
     * @var string $version
     */
    public  $version ;
    /**
     * Singleton Instance
     *
     * @var DMS
     */
    private static  $instance ;
    /**
     * Holds the flag if external host/domain is mapped
     *
     * @var bool
     */
    private  $domain_match = false ;
    /**
     * Current HTTP_HOST
     *
     * @var String
     */
    private  $domain ;
    /**
     * Current request uri path
     *
     * @var String
     */
    private  $path ;
    /**
     * Current request uri query string
     *
     * @var String
     */
    private  $query_string ;
    /**
     * Map of Host/Post ID
     *
     * @var String[int]
     */
    private  $map = array() ;
    /**
     * DMS constructor.
     */
    private function __construct()
    {
    }
    
    /**
     * Singleton
     *
     * @return DMS
     */
    public static function getInstance()
    {
        if ( !isset( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Define main used class properties
     *
     * @param   string    $pluginBaseName
     * @param   string    $pluginDir
     * @param   string    $pluginUrl
     * @param   string    $version
     * @param   Freemius  $dms_fs
     */
    public static function defineProperties(
        $pluginBaseName,
        $pluginDir,
        $pluginUrl,
        $version,
        $dms_fs
    )
    {
        $DMS = self::getInstance();
        $DMS->plugin_base_name = $pluginBaseName;
        $DMS->plugin_dir = $pluginDir;
        $DMS->plugin_url = $pluginUrl;
        $DMS->version = $version;
        $DMS->dms_fs = $dms_fs;
    }
    
    /**
     * Load plugin text domain
     *
     * @return void
     */
    public static function load_plugin_text_domain()
    {
        $DMS = self::getInstance();
        load_plugin_textdomain( 'domain-mapping-system', false, basename( $DMS->plugin_dir ) . '/languages' );
    }
    
    /**
     * Runs DMS, executed on WP init-Hook
     *
     * @return void
     */
    public static function run()
    {
        
        if ( !is_admin() ) {
            // Get instance
            $DMS = self::getInstance();
            // Retrieve and declare mapping stored in our side
            //TODO could be cached in future
            $DMS->generateMap();
            // Define requested domain/host
            $DMS->setCurrentDomain();
            $match = false;
            // Loop through out mappings and check for match
            foreach ( $DMS->map as $key => $data ) {
                $mapped_host = trim( $key, "/" );
                $current_url = trim( $DMS->domain, "/" );
                
                if ( $mapped_host === $current_url ) {
                    $match = true;
                    break;
                }
            
            }
            // If match, then check if path exists or no
            
            if ( $match ) {
                // Anyway we must store domain match flag
                $DMS->domain_match = true;
                // Check path emptiness
                
                if ( empty($DMS->path) && $DMS->dms_fs->can_use_premium_code__premium_only() ) {
                    // If empty, then we must show the primary mapping ( if premium is active )
                    $DMS->map( $data['primary'] );
                } elseif ( empty($DMS->path) && !$DMS->dms_fs->can_use_premium_code__premium_only() && !empty($data['values'][0]) ) {
                    // If empty but we plan is not premium, then we must show regular mapped page
                    $DMS->map( $data['values'][0] );
                }
            
            }
        
        }
    
    }
    
    /**
     * Save default config options at very first install
     * The first conditions is implemented by the 'dms_use_page' option existence
     *
     * @return void
     */
    public static function activate()
    {
        $DMS = self::getInstance();
        // Get page usage option
        $use_page = get_option( 'dms_use_page' );
        
        if ( $use_page === false ) {
            // Retrieve types if any
            $types = DMS::getCustomPostTypes();
            // Very first stage
            add_option( 'dms_use_page', 'on' );
            add_option( 'dms_use_post', 'on' );
            add_option( 'dms_use_categories', 'on' );
            foreach ( $types as $cpt ) {
                add_option( "dms_use_{$cpt['name']}", 'on' );
                if ( $cpt['has_archive'] ) {
                    add_option( "dms_use_{$cpt['name']}_archive", 'on' );
                }
            }
        } else {
            // Next activations case
            // Check for old style dms_map structure
            $dms_map = get_option( 'dms_map' );
            
            if ( !empty($dms_map) && !is_array( $dms_map ) ) {
                // Proper had to be version compare, but we dont have too much users yet
                // Is old string structure
                // Try to bring new to structure
                $new_dms_map = [];
                parse_str( $dms_map, $dms_map_arr );
                foreach ( $dms_map_arr as $key => $value ) {
                    $new_dms_map['domains'][] = array(
                        'host'     => str_replace( '_', '.', $key ),
                        'mappings' => array(
                        'values'  => [ $value ],
                        'primary' => $value,
                    ),
                    );
                }
                // Update option
                update_option( 'dms_map', $new_dms_map );
            }
        
        }
        
        //		// Deactivate other active version
        $plugin_base_name = $DMS->plugin_base_name;
        $plugin_dir_path = $DMS->plugin_dir;
        $free_plugin_base_name = ( strpos( $plugin_base_name, '-pro' ) === false ? $plugin_base_name : str_replace( basename( $plugin_dir_path ) . '-pro', basename( $plugin_dir_path ), $plugin_base_name ) );
        $premium_plugin_base_name = ( strpos( $plugin_base_name, '-pro' ) !== false ? $plugin_base_name : str_replace( basename( $plugin_dir_path ), basename( $plugin_dir_path ) . '-pro', $plugin_base_name ) );
        $is_premium_version_activation = current_filter() !== 'activate_' . $free_plugin_base_name;
        // This logic is relevant only to plugins since both the free and premium versions of a plugin can be active at the same time.
        // 1. If running in the activation of the FREE module, get the basename of the PREMIUM.
        // 2. If running in the activation of the PREMIUM module, get the basename of the FREE.
        $other_version_basename = ( $is_premium_version_activation ? $free_plugin_base_name : $premium_plugin_base_name );
        /**
         * If the other module version is active, deactivate it.
         *
         * is_plugin_active() checks if the plugin is active on the site or the network level and
         * deactivate_plugins() deactivates the plugin whether it's activated on the site or network level.
         *
         */
        if ( is_plugin_active( $other_version_basename ) ) {
            deactivate_plugins( $other_version_basename );
        }
    }
    
    /**
     * Unregister WP settings, executed on Plugin deactivation
     *
     * @return void
     */
    public static function deactivate()
    {
    }
    
    /**
     * Register DMS Settings and enqueue Scripts and Styles
     *
     * @return void
     */
    public static function adminInit()
    {
        $DMS = self::getInstance();
        $DMS->registerSettings();
        $DMS->startSession();
    }
    
    /**
     * Adds Admin Options Page
     *
     * @return void
     */
    public static function adminMenu()
    {
        $page = add_menu_page(
            __( 'Domain Mapping', 'domain-mapping-system' ),
            __( 'Domain Mapping', 'domain-mapping-system' ),
            'manage_options',
            'domain-mapping-system',
            array( 'DMS', 'includeTemplate' ),
            'dashicons-admin-site-alt3'
        );
        add_action( 'admin_print_styles-' . $page, array( 'DMS', 'registerStyles' ) );
        add_action( 'admin_print_scripts-' . $page, array( 'DMS', 'registerScripts' ) );
    }
    
    /**
     * Include DMS Option Template
     *
     * @return void
     */
    public static function includeTemplate()
    {
        $instance = self::getInstance();
        $dms_fs = $instance->dms_fs;
        if ( !current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have the permissions to access this page.' ) );
        }
        include_once $instance->plugin_dir . '/templates/option-page.php';
    }
    
    /**
     * Register WordPress Settings
     *
     * @return $this
     */
    private function registerSettings()
    {
        $types = self::getCustomPostTypes();
        register_setting( 'dms_storage', 'dms_enable_query_strings' );
        register_setting( 'dms_storage', 'dms_force_site_visitors' );
        register_setting( 'dms_storage', 'dms_global_mapping' );
        register_setting( 'dms_storage', 'dms_map', array(
            'type'              => 'array',
            'sanitize_callback' => array( $this, 'dms_map_sanitize' ),
        ) );
        register_setting( 'dms_config', 'dms_use_page' );
        register_setting( 'dms_config', 'dms_use_post' );
        register_setting( 'dms_config', 'dms_use_categories' );
        foreach ( $types as $cpt ) {
            register_setting( 'dms_config', "dms_use_{$cpt['name']}" );
            if ( $cpt['has_archive'] ) {
                register_setting( 'dms_config', "dms_use_{$cpt['name']}_archive" );
            }
        }
        return $this;
    }
    
    /**
     * Start session
     * Will be used mainly for showing error message to user
     */
    private function startSession()
    {
        if ( session_id() == '' ) {
            session_start();
        }
    }
    
    /**
     * Show admin notice in case of existence. Check based on session 'dms_admin_warning'
     */
    public function showAdminNotice()
    {
        
        if ( !empty($_SESSION['dms_admin_warning']) || !empty($_SESSION['dms_admin_success']) ) {
            
            if ( !empty($_SESSION['dms_admin_warning']) ) {
                $type = 'warning';
                $message = $_SESSION['dms_admin_warning'];
                unset( $_SESSION['dms_admin_warning'] );
            } else {
                $type = 'success';
                $message = $_SESSION['dms_admin_success'];
                unset( $_SESSION['dms_admin_success'] );
            }
            
            ?>
            <div class="notice notice-<?php 
            echo  $type ;
            ?> is-dismissible">
                <p><?php 
            _e( $message, 'sample-text-domain' );
            ?></p>
            </div>
			<?php 
        }
    
    }
    
    /**
     * DMS map sanitization
     *
     * @param   array  $dms_map
     *
     * @return mixed
     */
    public function dms_map_sanitize( $dms_map )
    {
        //TODO check SSL validation: If initial domain has , then mapped one need to have too
        $referer = wp_get_referer();
        // Validate data
        $valid_host = true;
        $empty_values = false;
        $valid_primary = true;
        $empty_rows = [];
        foreach ( $dms_map['domains'] as $key => $item ) {
            // Check emptiness
            
            if ( empty($item['host']) && empty($item['mappings']['values'][0]) ) {
                // Just ignore in case the fields are empty and save was clicked
                $empty_rows[] = $key;
                continue;
            }
            
            // Validate host
            $parsed_host = wp_parse_url( 'http://' . $item['host'], PHP_URL_HOST );
            
            if ( $parsed_host != $item['host'] || strpos( $item['host'], '.' ) === false ) {
                $valid_host = false;
                break;
            }
            
            // Check if mapping exists
            
            if ( empty($item['mappings']['values'][0]) ) {
                $empty_values = true;
                break;
            }
        
        }
        // Check validness of the host
        
        if ( !$valid_host ) {
            $_SESSION['dms_admin_warning'] = __( 'Invalid host specified', 'domain-mapping-system' );
            wp_safe_redirect( $referer );
            exit;
        }
        
        // Check mapped objects emptiness
        
        if ( $empty_values ) {
            $_SESSION['dms_admin_warning'] = __( 'Mappings are missing.', 'domain-mapping-system' );
            wp_safe_redirect( $referer );
            exit;
        }
        
        // Check primary object existence
        
        if ( !$valid_primary ) {
            $_SESSION['dms_admin_warning'] = __( 'Primary value is missing in main values', 'domain-mapping-system' );
            wp_safe_redirect( $referer );
            exit;
        }
        
        // Remove empty rows
        if ( !empty($empty_rows) ) {
            // Just remove keys with empty fields
            foreach ( $empty_rows as $key_to_remove ) {
                unset( $dms_map['domains'][$key_to_remove] );
            }
        }
        $_SESSION['dms_admin_success'] = __( 'Successfully saved.', 'domain-mapping-system' );
        return $dms_map;
    }
    
    /**
     * Register & enqueue CSS
     */
    public static function registerStyles()
    {
        $instance = self::getInstance();
        wp_register_style(
            'select2-css',
            $instance->plugin_url . 'assets/css/select2.min.css',
            array(),
            $instance->version,
            'all'
        );
        wp_enqueue_style( 'select2-css' );
        wp_register_style(
            'dms-css',
            $instance->plugin_url . 'assets/css/dms.css',
            array(),
            $instance->version,
            'all'
        );
        wp_enqueue_style( 'dms-css' );
    }
    
    /**
     * Register & enqueue JS
     */
    public static function registerScripts()
    {
        $instance = self::getInstance();
        // Fetch translations for js files
        $translations = (include_once $instance->plugin_dir . 'assets/js/localizations/js-translations.php');
        /**
         * Collect data to localize
         * translations for JS
         * premium flag
         */
        $dms_fs_data = array(
            'translations' => $translations,
            'is_premium'   => (int) $instance->dms_fs->can_use_premium_code__premium_only(),
        );
        // Register main js dependencies
        // Woo is using same js, that is why deregister first to avoid conflicts
        wp_deregister_script( 'select2' );
        wp_register_script(
            'select2',
            $instance->plugin_url . 'assets/js/select2.full.min.js',
            array( 'jquery' ),
            $instance->version
        );
        wp_register_script(
            'dms-js',
            $instance->plugin_url . 'assets/js/dms.js',
            array( 'jquery' ),
            $instance->version
        );
        wp_enqueue_script( 'select2' );
        wp_enqueue_script( 'dms-js' );
        // Include js data into dms-js
        wp_localize_script( 'dms-js', 'dms_fs', $dms_fs_data );
    }
    
    /**
     * Generate Host/Post ID Map, and store primary
     */
    private function generateMap()
    {
        $dms_map = get_option( 'dms_map' );
        
        if ( !empty($dms_map['domains']) ) {
            $domains = $dms_map['domains'];
            foreach ( $domains as $key => $map ) {
                $this->map[$map['host']]['values'] = $map['mappings']['values'];
                $this->map[$map['host']]['primary'] = ( !empty($map['mappings']['primary']) ? $map['mappings']['primary'] : null );
            }
        }
    
    }
    
    /**
     * Set current HTTP_HOST
     *
     * @return String
     */
    private function setCurrentDomain()
    {
        $url_parsed = ( !empty($_SERVER['REQUEST_URI']) ? wp_parse_url( $_SERVER['REQUEST_URI'] ) : null );
        // Store up query existence
        if ( !empty($url_parsed['query']) ) {
            $this->query_string = $url_parsed['query'];
        }
        // Store up uri path
        if ( !empty($url_parsed['path']) ) {
            $this->path = trim( $url_parsed['path'], '/' );
        }
        // Set up host/domain
        $this->domain = $_SERVER["HTTP_HOST"];
        if ( !empty($this->query_string) && !$this->dms_fs->can_use_premium_code__premium_only() ) {
            $this->domain = get_site_url();
        }
        return $this->domain;
    }
    
    /**
     * DMS Magic
     *
     * Checks if current host is set to a certain wordpress object (page/post/category/etc ...).
     * Call only for primary mapped object
     *
     * @param   mixed  $pageID
     */
    private function map( $pageID )
    {
        /*
         * If $pageID is numeric, it is a Page, Post or CPT ID.
         * Thus, we configure the query_post arguments to the single object.
         */
        
        if ( is_numeric( $pageID ) ) {
            $postType = get_post_type( $pageID );
            
            if ( $postType != 'page' ) {
                $args = array(
                    'post_type' => $postType,
                    'p'         => $pageID,
                );
            } else {
                $args = array(
                    'page_id' => $pageID,
                );
            }
            
            query_posts( $args );
        } else {
            /*
             * If $pageID is NOT numeric, it is a string like "archive_my_cpt".
             * Thus, we configure the query_post arguments to the archive.
             * Because of some strange WP behaviour, we need to include the CPT-Archive
             * TPL manually, and exit() after that.
             */
            
            if ( is_numeric( strpos( $pageID, 'archive' ) ) ) {
                $this->loadArchive( $pageID );
            } elseif ( is_numeric( strpos( $pageID, 'category' ) ) ) {
                $this->loadCategory( $pageID );
            }
        
        }
    
    }
    
    /**
     * Query CPT posts & load (CTP) Archive Templates
     *
     * Includes archive-{$cpt-type}.php and terminates.
     * If archive-Template is not present, WordPress fallback
     * is used.
     *
     * @param   String  $pageID
     */
    private function loadArchive( $pageID )
    {
        $postType = substr( $pageID, 8 );
        $args = array(
            'post_type'   => $postType,
            'm'           => 0,
            'p'           => 0,
            'post_parent' => 0,
        );
        query_posts( $args );
        /*
         * $file is the path for the CPT Archive Template
         */
        $file = TEMPLATEPATH . "/archive-{$postType}.php";
        /*
         * If a CPT Archive Template exists, use it and kill the script. Otherwise
         * let WordPress handle the fallback stuff.
         */
        
        if ( file_exists( $file ) ) {
            include_once $file;
            exit( 0 );
        }
    
    }
    
    /**
     * Query Posts by Category, let WP handle the rest.
     *
     * @param   mixed  $pageID
     */
    private function loadCategory( $pageID )
    {
        $category = substr( $pageID, 9 );
        $args = array(
            'category_name' => $category,
        );
        query_posts( $args );
    }
    
    /**
     * Get clean array of CPT
     *
     * @return array
     */
    public static function getCustomPostTypes()
    {
        $types = get_post_types( array(
            'public'   => true,
            '_builtin' => false,
        ), 'objects' );
        $cleanTypes = array();
        foreach ( $types as $item ) {
            $cleanTypes[] = array(
                'name'        => $item->query_var,
                'label'       => $item->labels->name,
                'has_archive' => $item->has_archive,
            );
        }
        return $cleanTypes;
    }
    
    /**
     * Force redirect in case flag is enabled. Will redirect to the mapped domain in case
     * requested object matches with the saved map item
     */
    public static function forceRedirect()
    {
        global  $wp_query ;
        $DMS = self::getInstance();
    }
    
    /**
     * Prevent redirection to base domain in case mapped domain requested without path.
     * Will work only if front page requested and mapped domain is connected with front page as primary
     *
     * @param   string  $redirectTo
     * @param   string  $redirectFrom
     *
     * @return string|false
     */
    public static function preventHomeRedirect( $redirectTo, $redirectFrom )
    {
        $DMS = self::getInstance();
        if ( $DMS->domain_match && is_front_page() ) {
            return false;
        }
        return $redirectTo;
    }
    
    /**
     * Runs when query variable object is ready , but before query is executed.
     * This way we can check if the queried object is mapped or no.
     * Note: this needs to work only if domain matches and path exists
     */
    public static function catchQueriedObject()
    {
        global  $wp_query ;
        $DMS = self::getInstance();
        
        if ( $DMS->domain_match && !empty($DMS->path) ) {
            // Check if queried object is mapped in our side
            $global_mapping_on = !empty(get_option( 'dms_global_mapping' ));
            $mappings = ( !empty($DMS->map[$DMS->domain]['values']) ? $DMS->map[$DMS->domain]['values'] : [] );
            
            if ( is_category() ) {
                $key = 'category-' . $wp_query->get_queried_object()->slug;
            } elseif ( is_single() || is_page() ) {
                $key = $wp_query->get_queried_object_id();
            } elseif ( is_post_type_archive() ) {
                //TODO
            }
            
            
            if ( empty($key) || $DMS->dms_fs->can_use_premium_code__premium_only() && $key == $DMS->map[$DMS->domain]['primary'] ) {
                //TODO check this earlier, also note that is_ssl will check current domain ssl,
                // and we are not sure about external website ssl
                $scheme = ( is_ssl() ? 'https://' : 'http://' );
                wp_redirect( $scheme . $DMS->domain . '/' . (( !empty($DMS->query_string) ? '?' . $DMS->query_string : '' )) );
                exit;
            } elseif ( !in_array( $key, $mappings ) && $global_mapping_on ) {
                // Do nothing
            } elseif ( !in_array( $key, $mappings ) ) {
                $wp_query->set_404();
                status_header( 404 );
            }
        
        }
    
    }
    
    /**
     * Compiles a clean list of DMS Options
     *
     * @return array
     */
    public static function getDMSOptions()
    {
        $posts = array();
        $usePages = get_option( 'dms_use_page' );
        
        if ( $usePages === 'on' ) {
            $pages = get_pages( array(
                'post_type' => 'page',
            ) );
            
            if ( !empty($pages) ) {
                $posts['Pages'] = array();
                foreach ( $pages as $page ) {
                    $posts['Pages'][] = array(
                        'id'    => $page->ID,
                        'title' => $page->post_title,
                    );
                }
            }
        
        }
        
        $usePosts = get_option( 'dms_use_post' );
        
        if ( $usePosts === 'on' ) {
            $blogPosts = get_posts( array(
                'numberposts' => -1,
            ) );
            
            if ( !empty($blogPosts) ) {
                $posts['Posts'] = array();
                foreach ( $blogPosts as $post ) {
                    $posts['Posts'][] = array(
                        'id'    => $post->ID,
                        'title' => $post->post_title,
                    );
                }
            }
        
        }
        
        $useCats = get_option( 'dms_use_categories' );
        
        if ( $useCats === 'on' ) {
            $cats = get_categories();
            
            if ( !empty($cats) ) {
                $posts['Blog Categories'] = array();
                foreach ( $cats as $cat ) {
                    $posts['Blog Categories'][] = array(
                        'title' => $cat->name,
                        'id'    => "category-{$cat->slug}",
                    );
                }
            }
        
        }
        
        $cleanTypes = self::getCustomPostTypes();
        if ( !empty($cleanTypes) ) {
            foreach ( $cleanTypes as $type ) {
                $useCPT = get_option( "dms_use_{$type['name']}" );
                
                if ( $useCPT === 'on' ) {
                    $args = array(
                        'post_type'      => $type['name'],
                        'posts_per_page' => -1,
                    );
                    $loop = new WP_Query( $args );
                    
                    if ( $loop->have_posts() ) {
                        $posts[$type['label']] = array();
                        if ( $type['has_archive'] ) {
                            $posts[$type['label']][] = array(
                                'title' => $type['label'] . " Archive",
                                'id'    => "archive-{$type['name']}",
                            );
                        }
                        while ( $loop->have_posts() ) {
                            $loop->the_post();
                            $posts[$type['label']][] = array(
                                'title' => get_the_title(),
                                'id'    => get_the_ID(),
                            );
                        }
                    }
                
                }
            
            }
        }
        return $posts;
    }

}